###################################################################################
#   Copyright (c) 2024 STMicroelectronics.
#   All rights reserved.
#   This software is licensed under terms that can be found in the LICENSE file in
#   the root directory of this software component.
#   If no LICENSE file comes with this software, it is provided AS-IS.
###################################################################################
"""
Utility functions to read the json file generated by the NPU/ATONN/NeuralART compiler
"""

import logging
import os
import json
from typing import Union, Any, Optional
from pathlib import Path

from .logging_utilities import print_table, get_print_fcts
from .exceptions import ParserNetworkError


#
# History
#   v0.1 - initial version - minimal reader
#

__title__ = 'NPU Utility - Generated network json reader'
__version__ = '0.1'
__author__ = 'STMicroelectronics'


def _compute_bw_mbs(n_rw: int, n_cycles: int, freq: int = 1000000000) -> float:
    """Compute BW (MB/s)"""
    bw_ = n_rw * freq / n_cycles if n_cycles else 0.0
    return bw_ / (1024 * 1024)


class CNpuNetworkJson():
    """Class to read the generated JSON file"""

    def __init__(self, filepath: Union[Path, str],
                 logger: Union[str, logging.Logger, None] = None,
                 c_name: str = 'network'):
        """Constructor"""

        if isinstance(logger, str) or logger is None:
            logger = logging.getLogger()

        self._logger = logger
        self._filepath = Path(filepath)
        cdts_ = [f'{c_name}.json', 'c_info.json', f'{c_name}_c_info.json']
        if os.path.isdir(self._filepath):
            for cdt_ in cdts_:
                json_fpath = self._filepath / cdt_
                if os.path.isfile(json_fpath):
                    self._filepath = json_fpath

        if not os.path.isfile(self._filepath):
            msg_ = f'CNpuNetworkJson: \'{self._filepath}\' is not valid, cdts={cdts_}'
            raise ParserNetworkError(msg_)

        logger.debug('')
        logger.debug('-> CNpuNetworkJson (v%s), file=\"%s\"', __version__, self._filepath)

        with open(self._filepath, encoding="utf-8") as file_handle:
            self._data = json.load(file_handle)

        # checking version
        ver_ = self._data.get("json_schema_version", None)
        if ver_ is None or ver_ != '2.0':
            msg_ = f'CNpuNetworkJson: Json schema is not valid: {ver_} instead 2.0 - {self._filepath}'
            raise ParserNetworkError(msg_)

        logger.debug('JSON schema version : %s', ver_)

        graphs_ = self._data['graphs']
        nodes_ = {node['id']: node for node in graphs_[0]['nodes']}
        for node_ in nodes_.values():
            n_name_ = 'EpochBlock_' + node_['name'].split('_')[-1]
            node_['name'] = n_name_

        self._epochs = {}
        self._n_ops: int = 0
        self._n_compute_cycles: int = 0
        self._n_max_cycles: int = 0

        # retreive the ops/compute cycles and max cycles by epoch
        pwr_estimates = self._data['power_estimates']
        for pwr_ in pwr_estimates:
            name_ = nodes_[pwr_["node_id"]]["name"]
            map_ = nodes_[pwr_["node_id"]]["mapping"]
            self._n_ops += int(pwr_["ops"])
            self._n_compute_cycles += int(pwr_["compute_cycles"])
            self._n_max_cycles += int(pwr_["max_cycles"])
            item = {
                'mapping': map_,
                'ops': int(pwr_["ops"]),
                'compute_cycles': int(pwr_["compute_cycles"]),
                'max_cycles': int(pwr_["max_cycles"]),
                # 'node': nodes_[pwr_["node_id"]],
                'mem_accesses': {}
            }
            self._epochs[name_] = item

        # retreive the memory accesses by epoch and by mempool
        mpools_ = {mpool['id']: mpool for mpool in self._data['memory_pools']}
        mem_accesses_ = self._data['memory_accesses']

        for mem_acc in mem_accesses_:
            if mem_acc["mpool_id"] not in mpools_:
                continue
            node_name_ = nodes_[mem_acc["node_id"]]["name"]
            mpool_name_ = mpools_[mem_acc["mpool_id"]]["name"]
            m_cycles_ = self._epochs[node_name_]['max_cycles']
            # m_cycles_ = mem_acc['read_cycles'] + mem_acc['write_cycles']
            n_rw_ = mem_acc['reads'] + mem_acc['writes']
            bw_ = _compute_bw_mbs(n_rw_, m_cycles_)
            val_ = (mem_acc['reads'], mem_acc['read_cycles'],
                    mem_acc['writes'], mem_acc['write_cycles'], bw_)
            self._epochs[node_name_]['mem_accesses'][mpool_name_] = val_

        # log a short description
        self.summary(logger.debug)
        logger.debug('')

        msg_ = f'<- {str(self)}'
        logger.debug(msg_)

    @property
    def file_path(self):
        """."""
        return self._filepath

    def get_perf(self, name: str):
        """."""
        return self._epochs.get(name, {})

    def summary(self, logger: Optional[Union[str, logging.Logger, Any]] = None):
        """."""

        pr_fn, _ = get_print_fcts(self._logger, logger)

        pr_fn('')
        pr_fn(str(self))
        pr_fn('')

        header_ = ['name', 'mapping', 'n_ops', 'n_cycles', 'r/w cycles', 'max cycles',
                   'n_ops/cycle C-M-R',
                   'r/w (Bytes)', 'BW (MB/s)', 'mpool:BW (MB/s) [r/w bytes]']
        rows_ = []
        for k, v in self._epochs.items():
            mem_accesses_ = v['mem_accesses']
            n_rw_ = 0
            n_rw_cycles_ = 0
            bw_per_mpool = ''
            for key, m_a_ in mem_accesses_.items():
                n_rw_ += (m_a_[0] + m_a_[2])
                n_rw_cycles_ += (m_a_[1] + m_a_[3])
                if m_a_[4]:
                    bw_per_mpool += f'{key}:{m_a_[4]:.3f} [{m_a_[0]}/{m_a_[2]}] '
            cycles_ratio_ = v["compute_cycles"] / v["max_cycles"] if v["max_cycles"] else 0.0
            bw_ = _compute_bw_mbs(n_rw_, v["max_cycles"])
            op_per_cycles_ = v["ops"] / v["compute_cycles"] if v["compute_cycles"] else 0.0
            m_op_per_cycles_ = v["ops"] / v["max_cycles"] if v["max_cycles"] else 0.0
            perc_m_cycles = v["max_cycles"] * 100 / self._n_max_cycles
            row_ = [k, v["mapping"], f'{v["ops"]:,}', f'{v["compute_cycles"]:,}',
                    f'{n_rw_cycles_:,}',
                    f'{v["max_cycles"]:,} {perc_m_cycles:6.2f}%',
                    f'{op_per_cycles_:5.1f}  {m_op_per_cycles_:5.1f}  {cycles_ratio_:4.2f}',
                    f'{n_rw_:,}', f'{bw_:.3f}', bw_per_mpool
                    ]
            rows_.append(row_)
        title_ = 'JSON info - mapping, memory accesses and cycles per epoch'
        colalign_ = ('left', 'left', 'right', 'right', 'right', 'right',
                     'right',
                     'right', 'right', 'left')
        print_table(header_, rows_, pr_fn, title=title_, colalign=colalign_)

    def __str__(self):
        """."""
        op_per_cycles_ = self._n_ops / self._n_compute_cycles
        m_op_per_cycles_ = self._n_ops / self._n_max_cycles
        msg_ = f'CNpuNetworkJson: {len(self._epochs)} n_ops={self._n_ops}'\
               f' cycles={self._n_compute_cycles} m_cycles={self._n_max_cycles}'\
               f' ops/cycle ideal/min={op_per_cycles_:.1f}/{m_op_per_cycles_:.1f}'
        return msg_
