###################################################################################
#   Copyright (c) 2024 STMicroelectronics.
#   All rights reserved.
#   This software is licensed under terms that can be found in the LICENSE file in
#   the root directory of this software component.
#   If no LICENSE file comes with this software, it is provided AS-IS.
###################################################################################
"""
Utility functions to parse a network.c generated by the NPU/ATONN/NeuralART compiler
"""

import logging
import os
from sys import maxsize
from enum import Enum
import re
from abc import ABC
from typing import List, Union, Tuple, TextIO, NamedTuple, Any, Optional, Dict
from pathlib import Path

from .logging_utilities import print_table
from .exceptions import ErrorException, ParserNetworkError
from .json_reader import CNpuNetworkJson

#
# History
#
#   v0.1 - initial version
#   v0.2 - adding parsing of .mem_shape/.mem_ndims LL buffer attrs
#          improve base_address extraction
#


__title__ = 'NPU Utility - Generated network C-file parser'
__version__ = '0.2'
__author__ = 'STMicroelectronics'

TypeBaseAddress = Union[str, int]
TypeAddress = Tuple[TypeBaseAddress, int]


class EpochType(Enum):
    """Epoch type"""
    NONE = 0
    HW = 1
    SW = 2
    HYBRID = 3
    BLOB = 4

    @staticmethod
    def from_flags(flags: Union[str, List[str]]):
        """."""
        if isinstance(flags, str):
            flags = [flags]
        if 'EpochBlock_Flags_blob' in flags:
            return EpochType(4)
        if 'EpochBlock_Flags_pure_hw' in flags:
            return EpochType(1)
        if 'EpochBlock_Flags_pure_sw' in flags:
            return EpochType(2)
        if 'EpochBlock_Flags_hybrid' in flags:
            return EpochType(3)
        return EpochType(0)


class PUnitType(Enum):
    """Processing Unit Type"""
    STR_ENG = 0
    ARITH_ACC = 1
    ACTIV_ACC = 2
    POOL_ACC = 3
    CONV_ACC = 4
    PROCESSOR = 5
    NULL = 6

    @staticmethod
    def from_name(name: str):
        """."""
        if 'STREAM_ENG' in name:
            return PUnitType(0)
        if 'ARITH_ACC' in name:
            return PUnitType(1)
        if 'ACTIV_ACC' in name:
            return PUnitType(2)
        if 'POOL_ACC' in name:
            return PUnitType(3)
        if 'CONV_ACC' in name:
            return PUnitType(4)
        if 'PROCESSOR' in name:
            return PUnitType(5)
        return PUnitType(6)


def _size_desc_to_int(size_desc: str) -> int:
    """Convert str to int (mempool desc)"""

    if '?' in size_desc:
        s_ = 0
    elif 'KB' in size_desc:
        ss_ = size_desc.strip().split(' ')[0]
        s_ = int(float(ss_) * 1024)
    elif 'MB' in size_desc:
        ss_ = size_desc.strip().split(' ')[0]
        s_ = int(float(ss_) * 1024 * 1024)
    elif 'B' in size_desc:
        ss_ = size_desc.strip().split(' ')[0]
        s_ = int(float(ss_))
    else:
        s_ = int(size_desc)
    return s_


def _size_int_to_desc(size: int, kb_only: bool = False) -> str:
    """Convert int to str"""
    if kb_only:
        return f'{size / 1024:.2f} KB'
    if 0 <= size < 1024:
        return f'{size} B'
    if 1024 <= size < 1024 * 1024:
        return f'{size / 1024:,.2f} KB'
    if size >= 1024 * 1024:
        return f'{size / (1024 * 1024):,.2f} MB'
    return f'invalid size {size}'


class AddressDesc(NamedTuple):
    """Class to handle the target address definition"""

    base: TypeBaseAddress = 0
    offset: int = 0

    def is_relative(self) -> bool:
        """Indicates if the address is relative"""
        return isinstance(self.base, str)

    def __str__(self):
        """Return formatted description"""
        if isinstance(self.base, str):
            return f'({self.base} + {self.offset})'
        return f'({hex(self.base)}UL + {self.offset})'


def _get_address(param: str, lines: str) -> TypeBaseAddress:
    """Retreive the address definition"""

    reg_desc = r'\.' + param

    # pattern = ".<param> = {(unsigned char *)(<base>)},"
    # pattern = ".<param> = {(unsigned char *)__LL_ATON_LIB_VIRTUAL_TO_PHYSICAL_ADDR(<base>)},"
    # pattern = ".<param> = {(unsigned char *)(<base>) /* Equivalent hex address = <base> */},"
    # pattern = ".<param> = {(unsigned char *)__LL_ATON_LIB_VIRTUAL_TO_PHYSICAL_ADDR((uintptr_t)<base>)},"

    reg0 = re.compile(reg_desc + r' = \{\(unsigned char \*\)\w{0,}\((?P<base>.*?)\).*?\},')
    reg1 = re.compile(reg_desc + r' = \{\(unsigned char \*\)\w{0,}\(\(uintptr_t\)(?P<base>.*?)\).*?\},')

    val = reg0.search(lines)  # extract base
    if val:
        base_ = val.group('base').strip()
        if base_.startswith('0x'):  # absolute base@
            base_ = base_.replace('UL', '')
            base_ = int(base_, 16)
        else:  # relative base@
            val1 = reg1.search(lines)
            if val1 or base_.startswith('(uintptr_t)'):
                if val1:
                    base_ = val1.group('base').strip()
                else:
                    base_ = base_.replace('(uintptr_t)', '')
                if base_.startswith('0x'):
                    base_ = base_.replace('UL', '')
                    base_ = int(base_, 16)
            else:
                raise ParserNetworkError(f'Unable to read the base@ from \'{base_}\'')
        return base_
    return 0


def _get_blob_address(lines: str) -> TypeBaseAddress:
    """Retreive the address of the blob"""

    # pattern = ".blob_address = (uintptr_t)(<addr>),"

    reg0 = re.compile(r'.blob_address = \{\(uintptr_t\)\((?P<base>.*?)\)\},')
    val = reg0.search(lines)  # extract base
    if val:
        return val.group('base').strip()
    return ''


def _get_address_from_param(param: str, lines: str) -> TypeAddress:
    """Retreive address definition (base + offset) from LL_Buffer_InfoTypeDef desc."""

    reg_desc = '.' + param

    # pattern = ".<param> = {((unsigned char *)(<base> + <offset>)},"

    reg0 = re.compile(reg_desc + r' = \{\(\(unsigned char \*\)\((?P<base>.*?)\s\+\s(?P<offset>\w+)\){2,3}\},')
    reg1 = re.compile(r'\((?P<label>\w+)\)')

    val = reg0.search(lines)  # extract base + off
    if val:
        base_ = val.group('base').strip()
        off_ = val.group('offset').strip()
        if base_.startswith('0x'):  # absolute base@
            base_ = base_.replace('UL', '')
            base_ = int(base_, 16)
        else:  # relative base@
            val = reg1.search(base_)  # extract c-label
            if val:
                base_ = val.group('label')
            else:
                raise ParserNetworkError(f'Unable to read the base@ from \'{base_}\'')
        off_ = int(off_)
        return base_, off_
    return 0, 0


def _get_c_field_value(field: str, section: str,
                       def_value: Union[str, None] = None,
                       endline: str = ',', preline: str = r'\.') -> Union[str, Any]:
    """Extract field value (str format)"""
    # Expected patterns in section string
    #   .<field> = <value>,
    #   .<field> = "<value>",
    reg_desc = preline + field
    pattern = re.compile(reg_desc + r'\s{0,}\=\s{0,}\"{0,1}(?P<field>.*?)\"{0,1}' + endline)
    val = pattern.search(section)
    if not val:
        pattern = re.compile(reg_desc + r'\s{0,}\=\s{0,}\"(?P<field>.*?)\"' + endline)
        val = re.search(pattern, section)
    if val:
        res = val['field'].strip()
        res = res[1:] if res.startswith('"') else res
        res = res[:-1] if res.endswith('"') else res
        return res
    return def_value


class CompilerDesc(NamedTuple):
    """Class to handle the used ATONN/NeuralART(tm) compiler options"""

    desc: str = ''
    sha: str = ''
    build_date: str = ''
    csv_file: str = ''
    all_buffers_info: bool = False
    model_file: str = ''
    json_quant_file: str = ''
    mempool_file: str = ''
    c_name: str = ''
    epoch_ctrl: bool = False
    version: Tuple[int] = (0, 0, 0, 0)

    def str_version(self):
        """."""
        return f'{self.desc} {self.version}'

    def __str__(self):
        return f'{self.desc} {self.version} {self.build_date}'


def parse_compiler_options(lines: Union[TextIO, List[str]],
                           logger: Optional[Union[str, logging.Logger]] = None) -> CompilerDesc:
    """Parse the compiler informations"""

    def _get_last_item(line: str):
        """Return the last item"""
        items = line.strip().split(' ')
        return items[-1].replace('"', '')

    if isinstance(logger, str) or logger is None:
        logger = logging.getLogger()

    if not isinstance(lines, list):
        lines.seek(0)
        lines = lines.readlines()

    msg_ = f'-> Parsing the compiler options.. {len(lines)}'
    logger.debug(msg_)

    desc_, sha_, date_, csv_file_ = '', '', '', ''
    all_buffers_info_, model_file_, json_quant_file_, mempool_ = False, '', '', ''
    ec_ = False
    version_ = (0, 0, 0, 0)
    c_name_ = ''
    for line in lines:
        if 'GIT_DESCRIPTION' in line:
            desc_ = _get_last_item(line)
        elif 'GIT_SHA' in line:
            sha_ = _get_last_item(line)
        elif 'BUILD_DATE' in line:
            date_ = _get_last_item(line)
        elif '--csv-file' in line:
            csv_file_ = _get_last_item(line)
        elif '--all-buffers-info' in line:
            all_buffers_info_ = _get_last_item(line) == 'true'
        elif '--onnx-input' in line:
            file_path_ = _get_last_item(line).replace('\\', '::').replace('/', '::')
            model_file_ = file_path_.split('::')[-1]
        elif '--json-quant-file' in line:
            file_path_ = _get_last_item(line).replace('\\', '::').replace('/', '::')
            json_quant_file_ = file_path_.split('::')[-1]
        elif '--load-mpool-file' in line:
            file_path_ = _get_last_item(line).replace('\\', '::').replace('/', '::')
            mempool_ = file_path_.split('::')[-1]
        elif 'enable-epoch-controller' in line:
            ec_ = _get_last_item(line) == 'true'
        elif 'LL_ATON_VERSION_MAJOR' in line:
            line = line.replace('#if', '').replace('!=', '').replace('||', '')
            line = line.strip().split()
            version_ = (int(line[1]), int(line[3]), int(line[5]), int(line[7]))

        if "/* global pool " in line:
            break

    p_reg = re.compile(r'const EpochBlock_ItemTypeDef \*LL_ATON_EpochBlockItems\_(?P<c_name>\w+)\(void\)*')
    for line in lines:
        line = line.strip()
        match = p_reg.match(line)
        if match:
            c_name_ = match.group('c_name')
            break

    options = CompilerDesc(desc_, sha_, date_,
                           csv_file_, all_buffers_info_, model_file_, json_quant_file_,
                           mempool_, c_name_, ec_, version_)

    logger.debug('<- done "%s"', str(options))

    return options


class MemoryPool():
    """Class to handle a memory pool descriptor"""

    def __init__(self, pool_id: int, pool_used_size: int, pool_desc: str):
        """Constructor"""
        self._id = pool_id
        assert int(pool_desc[0]) == pool_id
        self.used_size = pool_used_size

        postfix_ = pool_desc[1].strip()
        self._postfix = postfix_ if postfix_ else 'UNDEFINED'
        self._name = pool_desc[2]
        param_desc_ = 'offset=' + pool_desc[3]

        if 'absolute_mode' in param_desc_:
            self._mode = 'absolute'
        else:
            self._mode = 'relative'

        self._vpool: bool = 'vpool' in param_desc_

        param_ = _get_c_field_value('offset', param_desc_, '0x0', endline=' ', preline='')
        self._offset: int = int(param_, 16)

        param_ = _get_c_field_value('size', param_desc_, '0', endline=' ', preline='')
        self._size = int(param_)

        param_ = _get_c_field_value('cacheable', param_desc_, 'off', endline=' ', preline='')
        self.cacheable = param_.lower()
        if 'READ_WRITE' in param_desc_:
            self.attr = 'read/write'
        else:
            self.attr = 'read'

        param_ = _get_c_field_value('score', param_desc_, '0', endline=' ', preline='')
        self.score = int(param_)
        self._vid: Optional[int] = None
        self._mpools: List[int] = []
        self._c_label: Tuple[str, int] = ('', 0)
        self._used_addr_range = [maxsize, -maxsize - 1]
        self._used_param_addr_range = [maxsize, -maxsize - 1]

        self._buffer_counts: Tuple[List[str], List[str]] = ([], [])
        self._buffer_partially_counts: Tuple[List[str], List[str]] = ([], [])

    @property
    def pool_id(self) -> int:
        """Pool id"""
        return self._id

    @property
    def pool_vid(self) -> Optional[int]:
        """Return associated vpool id"""
        return self._vid

    def set_vid(self, pool_id):
        """Set associated vpool id"""
        self._vid = pool_id

    @property
    def mode(self) -> str:
        """Mode"""
        return self._mode

    @property
    def vpool(self) -> bool:
        """Virtual pool"""
        return self._vpool

    @property
    def mpools(self) -> List[int]:
        """Return list, id of the associated mpool"""
        return self._mpools

    @property
    def is_vpool(self) -> bool:
        """Is a virtual memory pool"""
        # assert self.vpool == bool(self._mpools)
        return bool(self._mpools)

    @property
    def is_relative(self) -> bool:
        """Return true if mode is relative"""
        return self._mode == 'relative'

    @property
    def is_absolute(self) -> bool:
        """Return true if mode is absolute"""
        return self._mode == 'absolute'

    @property
    def is_param_only(self) -> bool:
        """Return true if the mempool is only a param initializer"""
        norm_ = len(self._buffer_counts[0]) == 0 and len(self._buffer_counts[1]) > 0
        part_ = len(self._buffer_partially_counts[0]) == 0 and len(self._buffer_partially_counts[1]) > 0
        return norm_ or part_

    @property
    def is_rw(self) -> bool:
        """Return true if the mempool is a RW memory section"""
        return 'write' in self.attr

    @property
    def is_cacheable(self) -> bool:
        """Return true if the mempool is a cacheable memory section"""
        return 'on' in self.cacheable

    @property
    def with_params(self) -> bool:
        """Return true if the mempool contains the params"""
        return len(self._buffer_counts[1]) > 0 or len(self._buffer_partially_counts[1]) > 0

    @property
    def postfix(self) -> str:
        """Postfix descriptor"""
        if self.is_vpool:
            return ''
        return self._postfix

    @property
    def name(self) -> str:
        """Name descriptor"""
        if self.is_vpool:
            return 'vpool'
        return self._name

    @property
    def c_label(self) -> Tuple[str, int]:
        """Tuple with c-label and size"""
        return self._c_label

    @property
    def offset(self) -> int:
        """Base address/offset"""
        return self._offset

    @property
    def size(self) -> int:
        """Return size in bytes of the mempool"""
        return self._size

    def used(self, cat: str = 'total') -> int:
        """Return used size in bytes"""
        total_used_ = self._used_addr_range[1] - self._used_addr_range[0]
        total_size_ = self.used_size if total_used_ < 0 else total_used_ + 1
        param_used_ = self._used_param_addr_range[1] - self._used_param_addr_range[0]
        param_size_ = 0 if param_used_ < 0 else param_used_ + 1
        if cat == 'const':
            return param_size_
        elif cat == 'rw':
            return total_size_ - param_size_
        return total_size_

    def set_c_label(self, label):
        """."""
        self._c_label = (str(label[0]), int(label[1]))

    def add_sub_mempool(self, cdt):
        """."""
        if cdt.postfix in self._postfix.split('_'):
            self._vpool = True
            cdt.set_vid(self.pool_id)
            if not self.contains(cdt.offset, cdt.size, update_range=False):
                msg_ = f'Virtual mempool definition is invalid - id={cdt.pool_id} not in id={self.pool_id}'
                raise ParserNetworkError(msg_)
            if cdt.pool_id not in self._mpools:
                self._mpools.append(cdt.pool_id)
            return True
        return False

    def fix_vpool_id(self, mpools):
        """Fix mempool ids for vpool support"""
        for cdt in mpools:
            if cdt.pool_id == self.pool_id:
                continue
            cdt.add_sub_mempool(self)

    def contains(self, base_addr: int, size: int, update_range: bool = True, is_param: bool = False) -> int:
        """."""
        end_addr = base_addr + size - 1
        max_addr = self.offset + self.size - 1
        use_case = ''
        # Buffer not included in memory-pool
        if end_addr < self.offset or base_addr > max_addr or size <= 0:
            use_case = 'out'
            overl_size = 0
        # c0 - buffer included in the memory-pool
        elif base_addr >= self.offset and end_addr <= max_addr:
            if update_range:
                self._used_addr_range[0] = min(self._used_addr_range[0], base_addr)
                self._used_addr_range[1] = max(self._used_addr_range[1], end_addr)
                if is_param:
                    self._used_param_addr_range[0] = min(self._used_param_addr_range[0], base_addr)
                    self._used_param_addr_range[1] = max(self._used_param_addr_range[1], end_addr)
            use_case = 'c0'
            overl_size = size
        # c1 - buffer partially included in the memory-pool (right part)
        elif self.offset <= end_addr <= max_addr:
            if update_range:
                self._used_addr_range[0] = self.offset
                self._used_addr_range[1] = max(self._used_addr_range[1], end_addr)
                if is_param:
                    self._used_param_addr_range[0] = self.offset
                    self._used_param_addr_range[1] = max(self._used_param_addr_range[1], end_addr)
            use_case = 'c1'
            overl_size = end_addr - self.offset + 1
        # c2 - buffer partially included in the memory-pool (left part)
        elif self.offset <= base_addr <= max_addr:
            if update_range:
                self._used_addr_range[0] = min(self._used_addr_range[0], base_addr)
                self._used_addr_range[1] = max_addr
                if is_param:
                    self._used_param_addr_range[0] = min(self._used_param_addr_range[0], base_addr)
                    self._used_param_addr_range[1] = max_addr
            use_case = 'c2'
            overl_size = max_addr - base_addr + 1
        # c3 - memory-pool included in the buffer
        elif base_addr < self.offset and end_addr > max_addr:
            if update_range:
                self._used_addr_range[0] = self.offset
                self._used_addr_range[1] = max_addr
                if is_param:
                    self._used_param_addr_range[0] = self.offset
                    self._used_param_addr_range[1] = max_addr
            use_case = 'c3'
            overl_size = self.size
        else:
            use_case = '??'
            overl_size = 0

        if is_param and use_case != 'c0' and overl_size != 0:
            msg_ = f'Param buffer should be fully included in a the mempool {self.pool_id:02d}'\
                   f' s={overl_size:10d} buf={base_addr:08x}:{end_addr:08x} is_param={is_param} {use_case}'
            print(f"WARNING: {msg_}")  # noqa: T201
            # raise ParserNetworkError(msg_)

        return overl_size

    def add_buffer(self, is_param: int, name: str, partially: bool = False):
        """Add buffer"""
        if is_param:
            if partially:
                self._buffer_partially_counts[1].append(name)
            else:
                self._buffer_counts[1].append(name)
        else:
            if partially:
                self._buffer_partially_counts[0].append(name)
            else:
                self._buffer_counts[0].append(name)

    def n_buffers(self) -> Tuple[Tuple[int, int], Tuple[int, int]]:
        """."""
        return ((len(self._buffer_counts[0]), len(self._buffer_partially_counts[0])),
                (len(self._buffer_counts[1]), len(self._buffer_partially_counts[1])))

    def attrs_to_str(self) -> str:
        """Return human description of the attrs"""
        msg_ = 'rw' if 'write' in self.attr else 'ro'
        msg_ += '/c' if self.is_cacheable else '/-'
        msg_ += '/r' if self.is_relative else '/a'
        msg_ += '/v' if self.is_vpool else '/-'
        return msg_

    def __str__(self) -> str:
        """Description of the memory pool"""
        msg_ = f'MPOOL: {self.pool_id:2d} '
        msg_ += self.attrs_to_str()
        msg_ += f' 0x{self.offset:08x}({self.size / 1024:10.2f} KB)'
        msg_ += f' {_size_int_to_desc(self.used_size):>12s} :'
        msg_ += f' {_size_int_to_desc(self.used(), kb_only=True):>12s}'
        n_buffers_ = self.n_buffers()
        msg_ += f' - {n_buffers_[0][0]:4d}:{n_buffers_[0][1]:d}'
        msg_ += f' {n_buffers_[1][0]:4d}:{n_buffers_[1][1]:d}'
        msg_ += f' \'{self.name}\'/\'{self.postfix}\''
        if self.vpool:
            mpools_ = ' ,'.join([str(v) for v in self._mpools])
            if self._mpools:
                msg_ += f' uses {mpools_}'
        elif self._vid:
            msg_ += f' part of {self._vid}'
        if self.c_label[0]:
            msg_ += f' label=\'{self.c_label[0]}\''
        msg_ += ' (with params)' if self.with_params else ''

        return msg_

    __repr__ = __str__


def parse_mempools(lines: Union[TextIO, List[str]],
                   logger: Optional[Union[str, logging.Logger]] = None) -> List[MemoryPool]:
    """Parse the memory pool descriptors"""

    memory_pools: List[MemoryPool] = []

    if isinstance(logger, str) or logger is None:
        logger = logging.getLogger()

    if not isinstance(lines, list):
        lines.seek(0)
        lines = lines.readlines()

    logger.debug('-> Parsing the memory pool descriptors.. nblines=%s', len(lines))

    s_line = ''.join(lines)

    # /* global pool <index> is <size> */
    pool_id_sizes_ = re.findall(r'\/\* global pool\s(?P<index>[0-9]+)\s+is\s(?P<size>.*?)\s+\*\/', s_line)
    # /* index=<index> file postfix=<postfix> name=<name> offset=<desc> */
    pool_descs_ = re.findall(r'\/\* index=(?P<index>[0-9]+)\s+file '
                             + r'postfix=(?P<postfix>.*?)\sname=(?P<name>.*?)\s+offset=(?P<desc>.*?)\*\/',
                             s_line)
    # extern unsigned char _mem_pool_OCTOSPI1_Default[]; /* [303393]; */
    mempool_c_labels_ = re.findall(r'extern unsigned char (?P<name>.*?)\[\]\;\s+\/\*\s+\[(?P<size>.*?)\]', s_line)

    if len(pool_id_sizes_) != len(pool_descs_):
        raise ParserNetworkError('Number of pools and descriptors are not aligned')

    for pool_id_size, pool_desc in zip(pool_id_sizes_, pool_descs_):
        pool_id_ = int(pool_id_size[0])
        pool_used_size_ = _size_desc_to_int(pool_id_size[1])
        memory_pools.append(MemoryPool(pool_id_, pool_used_size_, pool_desc))
        logger.debug('adding %s', str(memory_pools[-1]))

    nb_mempool_relative = len(list(filter(lambda x: x.is_relative, memory_pools)))
    if len(mempool_c_labels_) != nb_mempool_relative:
        raise ParserNetworkError('Number of mempool relative and c-label are not aligned')

    # part of an virtual memory pool?
    c_label_idx = 0
    for mpool in memory_pools:
        if mpool.is_relative:
            mpool.set_c_label(mempool_c_labels_[c_label_idx])
            c_label_idx += 1
        mpool.fix_vpool_id(memory_pools)

    # filter/sort the mempools
    logger.debug('filtering/sorting the mpools')
    filtered_memory_pools = []
    for mpool in memory_pools:
        if mpool.size != 0 or mpool.offset != 0:
            filtered_memory_pools.append(mpool)

    filtered_memory_pools.sort(key=lambda a: a.offset, reverse=True)

    for mpool in filtered_memory_pools:
        logger.debug(str(mpool))

    logger.debug('<- %s memory pool descriptors found after filtering.', len(filtered_memory_pools))

    return filtered_memory_pools


class AddrDesc():
    """Class to handle an address descriptor (base + offset)"""

    def __init__(self, desc: TypeAddress):
        """Constructor"""
        self._base: Union[str, int] = desc[0]
        self._offset: int = desc[1]

    def __call__(self):
        """Base address"""
        return self._base + self._offset

    def get_base_and_offset(self):
        """Return tuple with base and offset"""
        return (self._base, self._offset)

    def __str__(self):
        """Return formatted str"""
        if isinstance(self._base, str):
            return f'({self._base} + {self._offset})'
        return f'({hex(self._base)}UL + {self._offset})'


class LLBufferDesc():
    """Class to handle a buffer descriptor (LL_Buffer_InfoTypeDef)"""

    def __init__(self, desc: str = '', category: str = '',
                 shapes: Optional[Dict] = None, scales: Optional[Dict] = None,
                 offsets: Optional[Dict] = None):

        self.category: str = category

        self.name: str = ''
        self.addr_start: TypeAddress = (0, 0)
        self.addr_end: TypeAddress = (0, 0)
        self.addr_limit: TypeAddress = (0, 0)
        self.is_param: bool = False
        self.is_user_allocated: int = 0
        self.epoch: int = -1
        self.batch: int = 1
        self.mem_shape: Union[str, List[int]] = ''
        self.mem_ndims: int = 0
        self.chpos: str = ''
        self.Qm: int = 7
        self.Qn: int = 0
        self.Qunsigned: int = 0

        self.type: str = 'DataType_INT8'
        self.nbits: int = 8
        self.ndims: int = 0
        self.per_channel: int = 0

        self.shape: Union[str, List[int]] = ''
        self.scale: Union[List[float], str, None] = None
        self.offset: Union[List[int], str, None] = None

        self.update(desc)

        if shapes:
            if self.ndims and self.shape:
                shape_ = shapes.get(self.shape, [0] * self.ndims)
                if self.ndims == 4:
                    self.shape = [shape_[0], shape_[3], shape_[1], shape_[2]]
                else:
                    self.shape = shape_
            if self.mem_ndims and self.mem_shape:
                shape_ = shapes.get(self.mem_shape, [0] * self.mem_ndims)
                self.mem_shape = shape_

        if scales and self.scale:
            self.scale = scales.get(self.scale, None)
        if offsets and self.offset:
            self.offset = offsets.get(self.offset, None)

        self.mpools: List[int] = []

        self.count = [0, 0]

    def update(self, desc: str):
        """Update the fields"""

        if not desc:
            return

        self.name = _get_c_field_value('name', desc, 'UNDEFINED').replace('"', '')

        offset_start_ = int(_get_c_field_value('offset_start', desc, '-1'))
        if offset_start_ >= 0:  # new format
            addr_base_ = _get_address('addr_base', desc)
            offset_end_ = int(_get_c_field_value('offset_end', desc, '0'))
            offset_limit_ = int(_get_c_field_value('offset_limit', desc, '0'))
            self.addr_start = (addr_base_, offset_start_)
            self.addr_end = (addr_base_, offset_end_)
            self.addr_limit = (addr_base_, offset_limit_)
        else:
            self.addr_start = _get_address_from_param('addr_start', desc)
            self.addr_end = _get_address_from_param('addr_end', desc)
            self.addr_limit = _get_address_from_param('addr_limit', desc)

        self.is_param = bool(int(_get_c_field_value('is_param', desc, '0')) == 1)
        self.is_user_allocated = int(_get_c_field_value('is_user_allocated', desc, '0'))
        self.epoch = int(_get_c_field_value('epoch', desc, '-1'))
        self.batch = int(_get_c_field_value('batch', desc, '0'))
        self.Qm = int(_get_c_field_value('Qm', desc, '0'))
        self.Qn = int(_get_c_field_value('Qn', desc, '0'))
        self.Qunsigned = int(_get_c_field_value('Qunsigned', desc, '0'))
        self.type = _get_c_field_value('type', desc, 'UNDEFINED')
        self.nbits = int(_get_c_field_value('nbits', desc, '0'))
        self.ndims = int(_get_c_field_value('ndims', desc, '0'))
        self.per_channel = int(_get_c_field_value('per_channel', desc, '0'))

        self.shape = _get_c_field_value('shape', desc, '')
        self.mem_shape = _get_c_field_value('mem_shape', desc, '')
        self.mem_ndims = int(_get_c_field_value('mem_ndims', desc, '0'))
        self.scale = _get_c_field_value('scale', desc, None)
        self.offset = _get_c_field_value('offset', desc, None)
        self.chpos = _get_c_field_value('chpos', desc, '')

    @property
    def is_relative(self):
        """Base@ is relative - represented by a c-label"""
        return isinstance(self.addr_start[0], str)

    @property
    def size(self):
        """."""
        return self.addr_end[1] - self.addr_start[1]

    def update_mempools(self, mempools: List[MemoryPool]):
        """."""
        for mempool in mempools:
            if self.is_relative:
                if mempool.c_label[0] == self.addr_start[0]:
                    s_addr = AddrDesc((mempool.offset, self.addr_start[1]))
                    e_addr = AddrDesc((mempool.offset, self.addr_end[1]))
                    size_ = mempool.contains(s_addr(), e_addr() - s_addr(), is_param=self.is_param)
                    if size_ != self.size:
                        msg_ = f'Buffer \'{self.name}\' is not contained in \'{mempool.name}\''
                        raise ParserNetworkError(msg_)
                    self.mpools.append(mempool.pool_id)
                    mempool.add_buffer(self.is_param, self.name)
                    break
            else:
                s_addr = AddrDesc(self.addr_start)
                e_addr = AddrDesc(self.addr_end)
                size_ = mempool.contains(s_addr(), e_addr() - s_addr(), is_param=self.is_param)
                if size_:
                    self.mpools.append(mempool.pool_id)
                    mempool.add_buffer(self.is_param, self.name, size_ != self.size)

    def addr_desc(self) -> str:
        """Return human readable short description of base addr"""
        addr_ = str(AddrDesc(self.addr_start))
        return addr_

    def short_desc(self) -> str:
        """Return human readable short description"""
        addr_ = str(AddrDesc(self.addr_start))
        desc_ = f'\'{self.name}\'/{addr_}/{self.size:,d}'
        return desc_

    def __str__(self):
        fmt_ = '{}b/Q{}.{} ({})'.format(self.nbits, self.Qm, self.Qn, self.Qunsigned) if self.Qm else f'{self.nbits}b'
        type_ = f'{self.type.replace("DataType_", "")} {fmt_}'
        addr_ = 'P ' if self.is_param else 'A '
        addr_ += self.short_desc()
        msg_ = f'BUFFER: {addr_} epoch={self.epoch}'
        shape_ = self.mem_shape if self.mem_shape else self.shape
        msg_ += f' {type_} {shape_} batch={self.batch} cat={self.category} mpools={self.mpools}'
        if not self.is_param:
            msg_ += f' pc={self.per_channel} {self.scale} {self.offset}'
        else:
            d_ = ''
            if self.scale:
                d_ = f'{len(self.scale)} {len(self.offset)}'
            msg_ += f' pc={self.per_channel} {d_}'
        return msg_


def parse_ll_buffers(lines: Union[TextIO, List[str]], mempools: List[MemoryPool],
                     logger: Optional[Union[str, logging.Logger]] = None
                     ) -> Tuple[List[LLBufferDesc], List[LLBufferDesc], List[LLBufferDesc]]:
    """Parse the list of the buffers"""

    buffer_descriptors: List[LLBufferDesc] = []
    buffer_inputs: List[LLBufferDesc] = []
    buffer_outputs: List[LLBufferDesc] = []

    if isinstance(logger, str) or logger is None:
        logger = logging.getLogger()

    if not isinstance(lines, list):
        lines.seek(0)
        lines = lines.readlines()

    logger.debug('-> Parsing the buffer lists.. nblines=%s', len(lines))

    line_descs: List[str] = []
    n_by_cat: Dict = {}
    cat_, name_ = '', ''
    shapes_: Dict = {}
    scales_: Dict = {}
    offsets_: Dict = {}
    for line in lines:
        line = line.strip()
        if not cat_:
            match = re.search(r'const\s+LL_Buffer_InfoTypeDef\s+\*LL_ATON_'
                              + r'(?P<cat>.*?)_Buffers_Info_(?P<name>.*?)(void)',
                              line)
            if match:
                cat_, name_ = match['cat'].lower(), match['name']
                n_by_cat[cat_] = [0, 0]
        else:
            # decode the shapes:  static const uint32_t buff_info__shape_1_64_96_1[] = { 1, 96, 1, 64 };
            match = re.search(r'static const uint32_t (?P<name>.*?)\[\] = \{(?P<dims>.*?)\}', line)
            if match:
                name_, dims_ = match['name'], match['dims']
                dims_ = dims_.strip().split(',')
                dims_ = [int(v_) for v_ in dims_]
                shapes_[name_] = dims_
            # decode the scales:  static const float buff_info_Input_0_out_0_quant_scale[] = { 0.00392156885936856 };
            match = re.search(r'static const float (?P<name>.*?)_scale\[\] = \{(?P<scales>.*?)\}', line)
            if match:
                name_, sc_ = match['name'], match['scales']
                sc_ = sc_.strip().split(',')
                sc_ = [float(v_) for v_ in sc_]
                scales_[name_ + '_scale'] = sc_
            # decode the offsets:  static const int16_t buff_info_Input_0_out_0_quant_offset[] = { -128 };
            match = re.search(r'static const int16_t (?P<name>.*?)_offset\[\] = \{(?P<offsets>.*?)\}', line)
            if match:
                name_, offs_ = match['name'], match['offsets']
                offs_ = offs_.strip().split(',')
                offs_ = [int(v_) for v_ in offs_]
                offsets_[name_ + '_offset'] = offs_
        if not line_descs and re.search(r'static const LL_Buffer_InfoTypeDef buff_info\[\] = \{', line):
            line_descs.append(line)
        elif line_descs and len(line) == 2 and '};' in line:
            # line_descs = ' '.join(line_descs)
            entries = re.findall(r'\{ (?P<instance>.*?) \}\,', ' '.join(line_descs))
            for entry in entries:
                buff_ = LLBufferDesc(entry, cat_, shapes_, scales_, offsets_)
                if buff_.is_param:
                    n_by_cat[cat_][1] += 1
                else:
                    n_by_cat[cat_][0] += 1
                buffer_descriptors.append(buff_)
                if 'input' in cat_ and buff_.is_param == 0:
                    buffer_inputs.append(buff_)
                if 'output' in cat_ and buff_.is_param == 0:
                    buffer_outputs.append(buff_)
            line_descs = []
            cat_, name_ = '', ''
            shapes_ = {}
        elif line_descs and line:
            line_descs.append(line)

    for buff_ in buffer_descriptors:
        buff_.update_mempools(mempools)
        logger.debug('%s', buff_)

    for buff_ in buffer_inputs:
        logger.debug('I: %s', buff_)
    for buff_ in buffer_outputs:
        logger.debug('O: %s', buff_)

    if logger.isEnabledFor(logging.DEBUG):
        logger.debug('-> Retreive the buffers mapped on different memory pools..')
        nb_buff = 0
        for buff in buffer_descriptors:
            if len(buff.mpools) > 2:
                nb_buff += 1
                logger.debug(buff)
        msg_ = f'<- {nb_buff} buffer(s) found'
        logger.debug(msg_)

    logger.debug('-> Mempool contents')
    for mpool in mempools:
        logger.debug(mpool)
    logger.debug('<-')

    logger.debug('<- %s buffer(s) found. (%s)', len(buffer_descriptors), n_by_cat)

    return buffer_inputs, buffer_outputs, buffer_descriptors


class ProcessingUnit(ABC):
    """Base class to define the Processing Units"""

    def __init__(self, unit: int, name: str, idx: int):

        self._putype: PUnitType = PUnitType.from_name(name)
        self.unit = int(unit)
        self.name = name
        self.id = int(idx)

        self.kind = ''
        self.node = ''

        self.is_omitted = False

    @property
    def putype(self):
        """Return PU type"""
        return self._putype

    def _set_attr(self, **kwargs):
        pass

    def set_attr(self, **kwargs):
        """Set the attributs"""
        self._set_attr(**kwargs)
        self.node = kwargs.pop('node', self.node)
        self.kind = kwargs.pop('kind', self.kind)

    def __str__(self):
        msg_ = f'{self.putype.name}.{self.id}'
        if self.kind:
            msg_ += f'.{self.kind}'
        if self.node:
            msg_ += f' node=\'{self.node}\''
        return msg_


class StreamEngineUnit(ProcessingUnit):
    """StreamEngine PU"""  # STREAM_ENG_V2

    def __init__(self, unit: int, name: str, idx: int):
        self.io_type = ''  # IO type
        self.port: int = -1  # port ID of the processing unit
        self.mempool: int = -1  # pool id
        self.addr_range = (0, 0)  # address range in the associated memory-pool
        super(StreamEngineUnit, self).__init__(unit, name, idx)

    @property
    def is_input(self):
        """."""
        return 'input' in self.io_type

    @property
    def size(self):
        """Return address range"""
        return self.addr_range[1] - self.addr_range[0]

    def _set_attr(self, **kwargs):
        self.io_type = kwargs.pop('io_type', self.io_type)
        self.node = kwargs.pop('node', self.node)
        self.port = int(kwargs.pop('port', self.port))
        if 'input' in self.io_type:
            self.kind = 'DmaIn'
            self.node += '_IN'
        else:
            self.kind = 'DmaOut'
            self.node += '_OUT'
        range_ = kwargs.pop('range', '')
        if range_:
            range_ = range_.replace('[', ' ').replace(']', ' ').replace(',', ' ')
            range_ = range_.strip().split(' ')             # range=5[20576,21360]
            self.mempool = int(range_[0])                  # 5
            self.addr_range = (int(range_[1]), int(range_[2]))  # (20576,21360)

    def __str__(self):
        return f'{self.putype.name}.{self.id}.{self.kind} {self.mempool}:{self.size}'

    __repr__ = __str__


def parse_processing_unit(lines: List[str]) -> List[ProcessingUnit]:
    """Parsing processing units (comments)"""

    # /* Unit= 0 [CONV_ACC_V2 0] */
    reg_unit = re.compile(r'\/\* Unit=\s+(?P<unit>[0-9]+)\s+\[(?P<name>.*?)\s(?P<id>.*?)\]\s\*\/')
    # /* kind=Mul node=Conv2D_3_mul_scale_3_in_36_2 */
    reg_kind = re.compile(r'\/\* kind=(?P<kind>.*?)\s+node=(?P<node>.*?)\s+\*\/')
    # /* Emit conf for STREAM_ENG_V2 node=MaxPool_7_in_55_2 output ports=0 range=5[28184,28860] */
    reg_strng = re.compile(r'\/\* Emit conf for STREAM_ENG_V2 node=(?P<node>.*?)\s+(?P<io_type>input|output)\s+'
                           r'ports=(?P<ports>.*?)\s+range=(?P<range>.*?)\s+\*\/')
    # /* node=Gemm_9_reshape_x_3 satisfies input and output adjacency (DMA->DMA) and can be omitted */
    reg_no_omitted = r' satisfies input and output adjacency \(DMA->DMA\) and can be omitted\s+\*\/'

    processing_units: List[ProcessingUnit] = []
    iter_list = iter(lines)
    p_unit: Union[StreamEngineUnit, ProcessingUnit]
    for line in iter_list:
        res = reg_unit.search(line.strip())
        if res:
            next_line = next(iter_list)
            pu_ = PUnitType.from_name(res['name'])
            unit_ = int(res['unit'])
            id_ = int(res['id'])
            if pu_ == PUnitType.STR_ENG:
                p_unit = StreamEngineUnit(unit_, res['name'], id_)
                res = reg_strng.search(next_line)
                if res is not None:
                    p_unit.set_attr(node=res['node'], io_type=res['io_type'],
                                    port=res['ports'], range=res['range'])
            else:
                p_unit = ProcessingUnit(unit_, res['name'], id_)
                res = reg_kind.search(next_line)
                if res is not None:
                    p_unit.set_attr(node=res['node'], kind=res['kind'])
                    next_line = next(iter_list)
                    res = re.search(f'/* node={p_unit.node}' + reg_no_omitted, next_line)
                    if res is not None:
                        p_unit.is_omitted = True
            processing_units.append(p_unit)

    return processing_units


class EpochFunctionDesc():
    """Class to handle the description of the Start/End functions"""

    def __init__(self, name: str, desc: Optional[List[str]] = None):
        self._name = name
        self._units: List[ProcessingUnit] = []
        if desc is not None:
            self._units = parse_processing_unit(desc)

    @property
    def name(self) -> str:
        """Return name of the function"""
        return self._name

    @property
    def units(self) -> List[ProcessingUnit]:
        """Return list of associated PUs"""
        return self._units

    @property
    def is_start(self) -> bool:
        """Start function"""
        return '_start_' in self.name.lower()

    def get_ops(self):
        """Return list of operations"""
        ops_ = []
        dins_ = [v.node for v in self._units if v.putype == PUnitType.STR_ENG and v.is_input]
        douts_ = [v.node for v in self._units if v.putype == PUnitType.STR_ENG and not v.is_input]
        for unit in self._units:
            if unit.putype == PUnitType.PROCESSOR:
                ops_.append(f'sw.{unit.kind.lower()}')
            elif unit.putype == PUnitType.NULL:
                if unit.node in dins_ and unit.node in douts_:
                    ops_.append(f'hw.memcpy.{unit.kind.lower()}')
            else:
                # if unit.putype != PUnitType.STR_ENG:
                ops_.append(f'hw.{unit.kind.lower()}')
        return ops_

    def units_to_dict(self):
        """Return dict with the used PU"""
        units_ = {}
        for unit in self.units:
            if unit.putype.name in units_:
                units_[unit.putype.name] += 1
            else:
                units_[unit.putype.name] = 1
        return units_

    def ops_to_dict(self):
        """Return dict with kind of operations"""
        kind_ops_ = {}
        for op_ in self.get_ops():
            if op_ in kind_ops_:
                kind_ops_[op_] += 1
            else:
                kind_ops_[op_] = 1
        return kind_ops_

    def __str__(self):
        units_ = {}
        for unit in self.units:
            if unit.putype.name in units_:
                units_[unit.putype.name] += 1
            else:
                units_[unit.putype.name] = 1
        return f'FCT:{self._name} {units_}'


_EPOCH_FCT_NULL = EpochFunctionDesc('')


def parse_epoch_start_end_functions(lines: Union[TextIO, List[str]],
                                    logger: Union[str, logging.Logger, None] = None
                                    ) -> Dict[str, EpochFunctionDesc]:
    """Parse the epoch start and end functions"""

    func_descriptors = {}

    if isinstance(logger, str) or logger is None:
        logger = logging.getLogger(logger)

    if not isinstance(lines, list):
        lines.seek(0)
        lines = lines.readlines()

    logger.debug('-> Parsing the start/end epoch functions.. nblines=%s', len(lines))

    reg0 = re.compile(r'static void LL_ATON_(?P<entry>Start|End)_EpochBlock_(?P<n0>\d+)\(const void \*epoch_block\)')

    # Extract lines with the start/end epoch functions
    line_descs: List[str] = []
    name_, n0_ = '', 0
    first_line_idx = -1
    for idx, line in enumerate(lines):
        line = line.strip()
        if not line_descs:
            match = reg0.search(line)
            if match:
                name_, n0_ = match['entry'], int(match['n0'])
                name_ = f'LL_ATON_{name_}_EpochBlock_{n0_}'
                line_descs.append(line)
                first_line_idx = idx + 1
        elif line_descs and len(line) == 1 and '}' in line and len(lines[idx + 1].strip()) == 0:
            line_descs.append(line)
            msg_ = f'-> parsing {name_} (line {first_line_idx} to {idx + 1})'
            logger.debug(msg_)
            if name_ in func_descriptors:
                raise ParserNetworkError("Start/Stop function %s already parsed" % name_)
            func_ = EpochFunctionDesc(name_, line_descs)
            for unit in func_.units:
                logger.debug(str(unit))
            func_descriptors[name_] = func_
            logger.debug('<- done - %s', str(func_))
            line_descs = []
            name_, n0_ = '', 0
        elif line_descs and line:
            line_descs.append(line)

    msg_ = f'<- {len(func_descriptors)} start/end epoch functions found'
    logger.debug(msg_)

    return func_descriptors


class EpochBlockDesc():
    """Class to handle the descrition of an epoch (EpochBlock_ItemTypeDef)"""

    def __init__(self, c_idx: int = -1, desc: str = ''):
        self.idx: int = c_idx
        self.start_epoch_block: EpochFunctionDesc = _EPOCH_FCT_NULL
        self.end_epoch_block: EpochFunctionDesc = _EPOCH_FCT_NULL
        self.blob_address: TypeBaseAddress = ''
        self.wait_mask: int = 0
        self.flags: List[str] = []
        self.epoch_num: int = -1
        self.last_epoch_num: int = -1
        self.in_streng_mask: Optional[int] = None
        self.out_streng_mask: Optional[int] = None
        self.perfs: Optional[Dict] = {}
        self.update(desc)

    @property
    def name(self):
        """Return name of the epoch"""
        return f'EpochBlock_{self.epoch_num}'

    @property
    def type(self):
        """Return type of the epoch"""
        return EpochType.from_flags(self.flags)

    def _decode_streng_mask(self, streng_mask) -> List[int]:
        """Decode the used STRENG ID"""
        streng_id: List[int] = []
        if streng_mask is None:
            return streng_id
        cur = streng_mask
        for idx in range(32):
            if cur & 0x1:
                streng_id.append(idx)
            cur = cur >> 1
        return streng_id

    def streng_in(self):
        """Return used Input STRENG"""
        return self._decode_streng_mask(self.in_streng_mask)

    def streng_out(self):
        """Return used Input STRENG"""
        return self._decode_streng_mask(self.out_streng_mask)

    def update(self, desc: str):
        """Parse the descriptor"""
        if not desc:
            return
        self.start_epoch_block = EpochFunctionDesc(_get_c_field_value('start_epoch_block', desc, ''))
        self.end_epoch_block = EpochFunctionDesc(_get_c_field_value('end_epoch_block', desc, ''))
        self.blob_address = _get_blob_address(desc)

        self.wait_mask = int(_get_c_field_value('wait_mask', desc, '0'), 16)
        flags_ = _get_c_field_value('flags', desc, '')
        self.flags = [v.strip() for v in flags_.split('|')]
        if 'LL_ATON_EB_DBG_INFO' in desc:
            self.epoch_num = int(_get_c_field_value('epoch_num', desc, '65535'))
            self.last_epoch_num = int(_get_c_field_value('last_epoch_num', desc, '0'))
            self.in_streng_mask = int(_get_c_field_value('in_streng_mask', desc, '0'), 16)
            self.out_streng_mask = int(_get_c_field_value('out_streng_mask', desc, '0'), 16)

    def is_last(self):
        """Indicates if the epoch is the last"""
        return 'EpochBlock_Flags_last_eb' in self.flags

    def get_streng_units(self) -> List[StreamEngineUnit]:
        """Return List of the used streng engines"""
        strg_units = [pu_ for pu_ in self.start_epoch_block.units if pu_.putype == PUnitType.STR_ENG]
        return strg_units

    def get_ops(self):
        """Return list with the kind of operations"""
        ops_ = self.start_epoch_block.get_ops()
        ops_ += self.end_epoch_block.get_ops()
        if self.type == EpochType.BLOB:
            return ['hw.blob']
        elif self.type == EpochType.HYBRID:
            return [v.replace('sw.', 'hybrid.') for v in ops_]
        return ops_

    def ops_to_dict(self):
        """Return dict with the kind of operations"""
        ops_ = self.start_epoch_block.ops_to_dict()
        e_ops_ = self.end_epoch_block.ops_to_dict()
        for key, val in e_ops_.items():
            if key not in ops_:
                ops_[key] = val
            else:
                ops_[key] += val
        return ops_

    def units_to_dict(self):
        """Return dict with the processing units"""
        units_ = self.start_epoch_block.units_to_dict()
        e_units_ = self.end_epoch_block.units_to_dict()
        for key, val in e_units_.items():
            if key not in units_:
                units_[key] = val
            else:
                units_[key] += val
        return units_

    def set_perf(self, values: Dict):
        """."""
        self.perfs = values

    def __str__(self):
        """Return formatted str"""
        flags_ = [flg.replace('EpochBlock_Flags_', '') for flg in self.flags]
        str_in_out_ = f'{self.streng_in()} -> {self.streng_out()}'
        start_fct = self.start_epoch_block.name
        end_fct = self.end_epoch_block.name
        return f'EPOCH: {self.idx:03d} {self.name} {start_fct}/{end_fct} {flags_} {str_in_out_} {self.type}'


def parse_epoch_block_list(lines: Union[TextIO, List[str]],
                           logger: Optional[Union[str, logging.Logger]] = None
                           ) -> List[EpochBlockDesc]:
    """Parse the list of the epoch block descriptor"""

    epoch_block_descriptors: List[EpochBlockDesc] = []

    if isinstance(logger, str) or logger is None:
        logger = logging.getLogger(logger)

    if not isinstance(lines, list):
        lines.seek(0)
        lines = lines.readlines()

    logger.debug('-> Parsing the list of the epoch descriptors.. nblines=%s', len(lines))

    # Extract lines with the EpochBlock descriptors
    line_descs: List[str] = []
    for line in lines:
        line = line.strip()
        if 'static const EpochBlock_ItemTypeDef ll_atonn_rt_epoch_block_array' in line:
            line_descs.append(line)
        elif line_descs and re.search(r'return\s+ll_atonn_rt_epoch_block_array', line):
            break
        elif line_descs and line:
            line_descs.append(line)
    line_descs.pop(0)
    line_descs.pop()

    # Decode the descriptors
    lines_: str = ' '.join(line_descs)
    entries = re.findall(r'\{ (?P<instance>.*?) \}\,', lines_)
    if entries:
        for idx, entry in enumerate(entries):
            desc = EpochBlockDesc(idx, entry)
            if not desc.is_last():
                epoch_block_descriptors.append(desc)

    # parse the start/stop functions
    start_stop_funcs = parse_epoch_start_end_functions(lines, logger)
    n_funcs = len(start_stop_funcs)

    for epoch in epoch_block_descriptors:
        if epoch.type != EpochType.BLOB:
            epoch.start_epoch_block = start_stop_funcs.pop(epoch.start_epoch_block.name,
                                                           _EPOCH_FCT_NULL)
            epoch.end_epoch_block = start_stop_funcs.pop(epoch.end_epoch_block.name,
                                                         _EPOCH_FCT_NULL)
        logger.debug('%s', str(epoch))
        logger.debug('  %s', epoch.get_ops())

    if len(start_stop_funcs):
        raise ParserNetworkError("%s func(s) not associated to a epoch descriptors" % len(start_stop_funcs))

    msg_ = f'<- {len(epoch_block_descriptors)} epoch descriptors found ({n_funcs})'
    logger.debug(msg_)

    return epoch_block_descriptors


class CNpuNetworkDesc():
    """Class to handle the deployed npu c-model file"""

    def __init__(self, filepath: Union[Path, str],
                 logger: Union[str, logging.Logger, None] = None,
                 c_name: str = 'network',
                 mem_only: bool = False):
        """Constructor"""

        if isinstance(logger, str) or logger is None:
            logger = logging.getLogger()

        self._options: CompilerDesc
        self._mpools: List[MemoryPool]
        self._buffers: List[LLBufferDesc]
        self._buffers_in: List[LLBufferDesc]
        self._buffers_out: List[LLBufferDesc]
        self._epochs: List[EpochBlockDesc] = []
        self._lines: List[str] = []

        self._filepath = Path(filepath)
        if os.path.isdir(self._filepath):
            self._filepath = Path(filepath).joinpath(f'{c_name}.c')

        if not os.path.isfile(self._filepath):
            msg_ = f'\'{self._filepath}\' is not a regular file'
            raise ParserNetworkError(msg_)

        logger.debug('')
        logger.debug('-> CNpuNetworkDesc (v%s), file=\"%s\"', __version__, self._filepath)

        with open(self._filepath, encoding="utf-8") as file_handle:
            self._options = parse_compiler_options(file_handle, logger=logger)
            self._mpools = parse_mempools(file_handle, logger=logger)
            self._buffers_in, self._buffers_out, self._buffers = parse_ll_buffers(file_handle,
                                                                                  self._mpools,
                                                                                  logger=logger)
            if not mem_only:
                self._epochs = parse_epoch_block_list(file_handle, logger=logger)
            file_handle.seek(0)
            self._lines = file_handle.readlines()

        json_object = CNpuNetworkJson(self._filepath.parent, logger)

        self._external = [0, 0]
        self._internal = [0, 0]
        for mpool in self._mpools:
            if mpool.is_vpool:
                continue
            if mpool.offset >= 0x60000000:
                self._external[1] += mpool.used('const')
                self._external[0] += mpool.used('rw')
            else:
                self._internal[1] += mpool.used('const')
                self._internal[0] += mpool.used('rw')

        self._epoch_type: Dict[str, int] = {}
        for epoch in self._epochs:
            if epoch.type in self._epoch_type:
                self._epoch_type[epoch.type] += 1
            else:
                self._epoch_type[epoch.type] = 1
            if json_object:
                if epoch.type == EpochType.BLOB:
                    # accumulate the perfs (num -> last_num)
                    perfs_ = {
                        'mapping': 'NODE_EC',
                        'ops': 0,
                        'compute_cycles': 0,
                        'max_cycles': 0,
                        'mem_accesses': {}
                    }
                    for num_id in range(epoch.epoch_num, epoch.last_epoch_num + 1):
                        cur_name_ = f'EpochBlock_{num_id}'
                        cur_perf_ = json_object.get_perf(cur_name_)
                        if cur_perf_['mapping'] == perfs_['mapping']:
                            perfs_['ops'] += cur_perf_['ops']
                            perfs_['compute_cycles'] += cur_perf_['compute_cycles']
                            perfs_['max_cycles'] += cur_perf_['max_cycles']
                            for key, val in cur_perf_['mem_accesses'].items():
                                if key in perfs_['mem_accesses']:
                                    val2 = perfs_['mem_accesses'][key]
                                    perfs_['mem_accesses'][key] = tuple(a + b for a, b in zip(val, val2))
                                else:
                                    perfs_['mem_accesses'][key] = val
                        else:
                            assert 0, f"Oh no! {cur_perf_['mapping']} != 'NODE_EC'"
                    epoch.set_perf(perfs_)
                else:
                    epoch.set_perf(json_object.get_perf(epoch.name))

        logger.debug('<- CNpuNetworkDesc done')
        self._logger = logger

    @property
    def filepath(self) -> Path:
        """Return used file-path"""
        return self._filepath

    @property
    def compiler(self) -> CompilerDesc:
        """Return compiler options"""
        return self._options

    @property
    def mpools(self) -> List[MemoryPool]:
        """Return mpool descriptors"""
        return self._mpools

    @property
    def mpools_by_id(self) -> Dict[int, MemoryPool]:
        """Return a dict with the mpool descriptors (key=pool_id)"""
        return {mpool_.pool_id: mpool_ for mpool_ in self._mpools}

    @property
    def mpools_by_name(self) -> Dict[str, MemoryPool]:
        """Return a dict with the mpool descriptors (key=pool_name)"""
        return {mpool_.name: mpool_ for mpool_ in self._mpools}

    @property
    def lines(self) -> List[str]:
        """Return c-lines"""
        return self._lines

    @property
    def buffers(self) -> List[LLBufferDesc]:
        """Return buffer descriptors"""
        return self._buffers

    @property
    def epochs(self) -> List[EpochBlockDesc]:
        """Return list of the epoch descriptors"""
        return self._epochs

    def get_epoch_desc(self, selector: Union[str, int]) -> Optional[EpochBlockDesc]:
        """Return specific epoch descriptor"""
        if not self.epochs:
            return None
        for epoch_desc in self.epochs:
            if isinstance(selector, int):
                if epoch_desc.epoch_num == selector:
                    return epoch_desc
            else:
                if epoch_desc.name == selector:
                    return epoch_desc
        return None

    def memories(self) -> Tuple[int]:
        """Return requested memories"""
        return (self._internal[0], self._internal[1], self._external[0], self._external[1])

    def _get_print_fcts(self, logger: Optional[Union[str, logging.Logger, Any]] = None, full: bool = False):
        """."""

        if isinstance(logger, str):
            logger_ = logging.getLogger(logger)
            print_fn = logger_.info
            print_deb_fn = self._logger.debug
        elif logger is None:
            print_fn = self._logger.info
            print_deb_fn = self._logger.debug
        elif isinstance(logger, logging.Logger):
            print_fn = logger.info
            print_deb_fn = logger.debug
        elif callable(logger):
            print_fn = logger
            print_deb_fn = None if not full else logger

        if full:
            print_deb_fn = print_fn

        return print_fn, print_deb_fn

    def _log_epochs(self, print_fn):
        """Log a table with the epoch descriptors"""

        if not self._epochs:
            return

        rows_ = []
        epoch_type = {}
        for epoch in self._epochs:
            ops_dict = {}
            if epoch.type in epoch_type:
                epoch_type[epoch.type] += 1
            else:
                epoch_type[epoch.type] = 1
            for op_ in epoch.get_ops():
                if op_ in ops_dict:
                    ops_dict[op_] += 1
                else:
                    ops_dict[op_] = 1
            ops_dict = ', '.join([f'{v}x {k}' for k, v in ops_dict.items()])
            ops_ = epoch.perfs.get('ops', '')
            if ops_:
                ops_dict = f'ops={ops_:,}, ' + ops_dict
            rows_.append([f'{epoch.name}', f'{epoch.type.name}', f'{ops_dict}'])
        epoch_type = ', '.join([f'{v}x {k.name}' for k, v in epoch_type.items()])

        header_ = ['epoch name', 'type', 'ops']
        title_ = f'epochs - {len(self._epochs)}: {epoch_type}'
        print_table(header_, rows_, print_fn, title=title_)

    def _log_mempools(self, print_fn):
        """Log a table with the mempool contents"""

        rows_ = []

        def _int_to_str(val):
            """."""
            val = '' if val <= 0 else _size_int_to_desc(val)
            return val

        for mpool in self._mpools:
            if not mpool.is_vpool:
                n_buffers_ = mpool.n_buffers()
                buffers_ = f'{n_buffers_[0][0]:4d}:{n_buffers_[0][1]:d}'
                buffers_ += f' {n_buffers_[1][0]:4d}:{n_buffers_[1][1]:d}'
                addr_ = mpool.c_label[0] if mpool.c_label[0] else f'{mpool.offset:08x}'
                name_ = f'{mpool.name} ([{mpool.pool_id}], {addr_}, {mpool.attrs_to_str()})'
                const_ = _int_to_str(mpool.used('const'))
                rw_ = _int_to_str(mpool.used('rw'))
                if mpool.offset >= 0x60000000:
                    rows_.append([name_, '', '', rw_, const_, buffers_])
                else:
                    rows_.append([name_, rw_, const_, '', '', buffers_])

        rows_.append(['', '', '', '', '', ''])
        rows_.append(['total', _size_int_to_desc(self._internal[0]), _size_int_to_desc(self._internal[1]),
                      _size_int_to_desc(self._external[0]), _size_int_to_desc(self._external[1])])

        const_ = f'{_size_int_to_desc(self._internal[1] + self._external[1])}'
        rw_ = f'{_size_int_to_desc(self._internal[0] + self._external[0])}'

        header_ = ['MPOOL ([id], addr, attrs)', 'data', 'const', 'data (ext)', 'const (ext)', '# buffers']
        colalign_ = ('left', 'right', 'right', 'right', 'right', 'right')
        title_ = f'mempools - act={rw_} params={const_}'
        print_table(header_, rows_, print_fn, colalign_, title=title_)

    def _log_io(self, print_fn):
        """Log a table with the IO descriptors"""
        rows_ = []
        sizes_ = [0, 0]

        for buff_ in self._buffers_in:
            addr_ = str(AddrDesc(buff_.addr_start))
            sizes_[0] += buff_.size
            shape_ = buff_.mem_shape if buff_.mem_shape else buff_.shape
            rows_.append([f'I: {buff_.name}{" (user)" if buff_.is_user_allocated else ""}',
                          f'{buff_.type[9:]} ({buff_.nbits}b)',
                          _size_int_to_desc(buff_.size), f'{shape_}', addr_])

        for buff_ in self._buffers_out:
            addr_ = str(AddrDesc(buff_.addr_start))
            sizes_[1] += buff_.size
            shape_ = buff_.mem_shape if buff_.mem_shape else buff_.shape
            rows_.append([f'O: {buff_.name}{" (user)" if buff_.is_user_allocated else ""}',
                          f'{buff_.type[9:]} ({buff_.nbits}b)',
                          _size_int_to_desc(buff_.size), f'{shape_}', addr_])

        header_ = ['IO TENSOR', 'type', 'size', 'shape', 'addr']
        colalign_ = ('left', 'left', 'right', 'right', 'left')
        title_ = f'io - {_size_int_to_desc(sizes_[0])}/{_size_int_to_desc(sizes_[1])}'
        print_table(header_, rows_, print_fn, colalign_, title=title_)

    def _log_buffer_type(self, print_fn):
        """Log a table with the type of buffers"""

        buffs_map = {}
        for buff_ in self._buffers:
            const_ = ' const' if buff_.is_param else ''
            item_name = buff_.type + f' ({buff_.nbits}b' + const_ + ')'
            if item_name in buffs_map:
                buffs_map[item_name][0] += 1
                buffs_map[item_name][1] += buff_.size
            else:
                buffs_map[item_name] = [1, buff_.size]

        rows_ = []
        for buff_type in buffs_map:
            rows_.append([buff_type[9:], buffs_map[buff_type][0], _size_int_to_desc(buffs_map[buff_type][1])])

        header_ = ['BUFFER type', '#', 'cumulated size']
        colalign_ = ('left', 'right', 'right')
        title_ = f'buffer type - {len(self.buffers)}'
        print_table(header_, rows_, print_fn, colalign_, title=title_)

    def _log_ops_type(self, print_fn):
        """."""
        if not self._epochs:
            return
        ops_hw_map = {}
        ops_hybrid_map = {}
        ops_sw_map = {}
        for epoch in self._epochs:
            for op_ in epoch.get_ops():
                if op_.startswith('hw.'):
                    ops_xx_ = ops_hw_map
                elif op_.startswith('hybrid.'):
                    ops_xx_ = ops_hybrid_map
                else:
                    ops_xx_ = ops_sw_map
                if op_ in ops_xx_:
                    ops_xx_[op_] += 1
                else:
                    ops_xx_[op_] = 1

        rows_ = []
        header_ = ['KIND of operation', '#']
        colalign_ = ('left', 'right')
        if ops_hw_map:
            for op in ops_hw_map:
                rows_.append([op, ops_hw_map[op]])
        if ops_hybrid_map:
            rows_.append(['', ''])  # SEPARATING_LINE)
            for op in ops_hybrid_map:
                rows_.append([op, ops_hybrid_map[op]])
        if ops_sw_map:
            rows_.append(['', ''])  # SEPARATING_LINE)
            for op in ops_sw_map:
                rows_.append([op, ops_sw_map[op]])

        title_ = f'op type - {len(ops_hw_map)} hw / {len(ops_hybrid_map)} hybrid / {len(ops_sw_map)} sw'
        print_table(header_, rows_, print_fn, colalign_, title=title_)

    def summary(self, logger: Optional[Union[str, logging.Logger, Any]] = None, full: bool = False):
        """."""

        pr_fn, pr_debug_fn = self._get_print_fcts(logger, full)

        pr_fn('')
        pr_fn(str(self))
        pr_fn('')
        msg_ = f' NPU compiler  : {self._options}\n'
        msg_ += f' model         : {self._options.model_file}\n'
        msg_ += f' json quant    : {self._options.json_quant_file}\n'
        msg_ += f' mempool       : {self._options.mempool_file}\n'
        msg_ += f' c_name        : {self._options.c_name}\n'
        msg_ += f' buffers_info  : {self._options.all_buffers_info}\n'
        msg_ += f' epoch ctrl    : {self._options.epoch_ctrl}\n'

        for line in msg_.splitlines():
            pr_fn(line)

        if pr_debug_fn and self.epochs:
            pr_debug_fn('')
            self._log_epochs(pr_debug_fn)
        if self.epochs:
            pr_fn('')
            self._log_ops_type(pr_fn)
        if pr_debug_fn:
            pr_debug_fn('')
            self._log_buffer_type(pr_debug_fn)
        pr_fn('')
        self._log_mempools(pr_fn)
        pr_fn('')
        self._log_io(pr_fn)

    def __str__(self):
        """."""
        epoch_type = ', '.join([f'{v}x {k.name}' for k, v in self._epoch_type.items()])
        const_ = f'({_size_int_to_desc(self._internal[1])}, {_size_int_to_desc(self._external[1])})'
        ram_ = f'({_size_int_to_desc(self._internal[0])}, {_size_int_to_desc(self._external[0])})'
        msg_ = f'CNpuNetworkDesc(v{__version__}):'
        if self._epochs:
            msg_ += f' {len(self.epochs)} epochs ({epoch_type}),'
        else:
            msg_ += ' <no epoch descriptors>,'
        msg_ += f' {len(self.buffers)} buffers'
        msg_ += f', ro={const_}, rw={ram_}'
        return msg_


def parse_c_network_file(args):
    """Helper fct to parse a generated network.c file"""

    from datetime import datetime

    logger = logging.getLogger()

    logger.info('%s (version %s)', __title__, __version__)
    logger.info('Creating date : %s', datetime.now().ctime())
    logger.info('')
    logger.info('Parsing the c-file : %s', args.cfile)

    c_npu_network = CNpuNetworkDesc(args.cfile, logger=logger)
    c_npu_network.summary(full=args.verbosity > 1)

    return 0


def main():
    """Script entry point."""

    import argparse
    import os
    import sys
    sys.path.insert(1, os.path.join(os.path.dirname(__file__)))

    from logging_utilities import get_logger

    _DEFAULT_NETWORK_FILE_PATH = 'model/model_kws_npu_ram/network.c'

    parser = argparse.ArgumentParser()

    parser.add_argument('--cfile', '-c', metavar='STR', type=str, help='generated c-file',
                        default=_DEFAULT_NETWORK_FILE_PATH)

    parser.add_argument('--cont', action='store_const', const=1,
                        help='Continue on error')

    parser.add_argument('--log', metavar='STR', type=str, nargs='?',
                        default='no-log',
                        help='log file')

    parser.add_argument('--verbosity', '-v',
                        nargs='?', const=1, type=int, choices=range(0, 3),
                        default=1, help="set verbosity level")

    parser.add_argument('--debug', action='store_const', const=1,
                        help='Enable internal log (DEBUG PURPOSE)')

    parser.add_argument('--no-color', action='store_const', const=1,
                        help='Disable log color support')

    args = parser.parse_args()

    lvl = logging.WARNING
    if args.verbosity > 0:
        lvl = logging.INFO
    if args.debug:
        lvl = logging.DEBUG

    if args.log is None:
        args.log = Path(__file__).stem + '.log'
    elif isinstance(args.log, str) and args.log == 'no-log':
        args.log = None

    logger = get_logger(level=lvl, color=not args.no_color, filename=args.log)

    try:
        res = parse_c_network_file(args)
    except ErrorException as e:
        logger.exception(e, stack_info=False, exc_info=args.debug)
        return -1

    return res


if __name__ == '__main__':
    import sys
    sys.exit(main())
