/****************************************************************************
*
* Copyright (c) 2025 STMicroelectronics - All Rights Reserved
*
* License terms: STMicroelectronics Proprietary in accordance with licensing
* terms SLA0098 at www.st.com.
*
* THIS SOFTWARE IS DISTRIBUTED "AS IS," AND ALL WARRANTIES ARE DISCLAIMED,
* INCLUDING MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE.
*
*****************************************************************************/
/**
 * @file    main.c
 * @brief   AI test the Multicore.
 *
 * The test runs the Stellar P3 cores simultaneously with AI networks.
 *
 * Pin configuration:
 *
 * UARTD5:
 * - TX PD[13] --> wired to serial port #1
 * - RX PD[12] --> wired to serial port #1
 *
 * UARTD1:
 *
 * On SR6X7-EVBC8000P (SR6P7G7) - SR6G7-EVBC6000P Rev A (SR6G7)
 * - TX PA[0] --> wire to JP17D (serial port #2)
 * - RX PD[8] --> wire to JP17C (serial port #2)
 * - LED blinking:
 *    - LED19 -> driven by Cortex-R52
 *    - LED20 -> driven by Cortex-M4
 *
 * On SR6PX-EVBC8000P Rev B (SR6P6 - SR6P7) - SR6G7-EVBC6000P Rev B (SR6G7)
 * - TX PE[7] --> wire to JP14D (serial port #2)
 * - RX PE[6] --> wire to JP14C (serial port #2)
 * - LED blinking:
 *    - LED20 -> driven by Cortex-R52
 *    - LED21 -> driven by Cortex-M4
 *
 * On SR6PX-EVBC4000P Rev A (SR6P3)
 * - TX PA[2] --> wire to JP14D (serial port #2)
 * - RX PD[8] --> wire to JP14C (serial port #2)
 * - LED blinking:
 *    - LED20 -> driven by Cortex-R52
 *    - LED21 -> driven by Cortex-M4
 *
 * On SR6G6-EVBC6000P Rev A (SR6G6)
 * - TX PA[2] --> wire to JP14D (serial port #2)
 * - RX PD[8] --> wire to JP14C (serial port #2)
 * - LED blinking:
 *    - LED25 -> driven by Cortex-R52
 *    - LED26 -> driven by Cortex-M4
 *
 * Serial ports configuration:
 * - 115200, N, 8, 1
 *
 * @{
 */

#include <stdio.h>
#include <string.h>
#include <stdlib.h>

#include <typedefs.h>

#include <io.h>
#include <irq.h>
#include <me.h>
#include <osal.h>
#include <siul2.h>
#include <uart.h>
#include <gst.h>

#include <clock.h>
#include <agt.h>
#include <linflexd.h>

#include <board_cfg.h>

/* DME/DSPH loading code/data */
#include <ld_sym.h>

/* R52 entry points */
#include <cluster0_core1_sym.h>
#ifdef sr6p3
#include <cluster1_core0_sym.h>
#elif defined(sr6g6)
#include <cluster2_core0_sym.h>
#include <cluster2_core1_sym.h>
#else
#include <cluster1_core0_sym.h>
#include <cluster1_core1_sym.h>
#include <cluster2_core0_sym.h>
#include <cluster2_core1_sym.h>
#endif

#include <board_cfg.h>
#include "stellar_ai_cfg.h"

#if (STELLAR_AI_VALIDATE == TRUE)
#include "stellar_ai.h"
#endif /* #if (STELLAR_AI_VALIDATE == TRUE) */

static void start_other_cores(void)
{
    /* Start cluster0 core1 */
    me_runcore(&MED0, ME_CORE_1_ID, __cluster0_core1_entry_point__);

#ifdef sr6p3
    /* Start cluster1 core0 */
    me_runcore(&MED0, ME_CORE_2_ID, __cluster1_core0_entry_point__);
#elif defined(sr6g6)
    /* Start cluster2 core0 */
    me_runcore(&MED2, ME_CORE_0_ID, __cluster2_core0_entry_point__);

    /* Start cluster2 core1 */
    me_runcore(&MED2, ME_CORE_1_ID, __cluster2_core1_entry_point__);
#else
    /* Start cluster1 core0 */
    me_runcore(&MED1, ME_CORE_0_ID, __cluster1_core0_entry_point__);

    /* Start cluster1 core1 */
    me_runcore(&MED1, ME_CORE_1_ID, __cluster1_core1_entry_point__);

    /* Start cluster2 core0 */
    me_runcore(&MED2, ME_CORE_0_ID, __cluster2_core0_entry_point__);

    /* Start cluster2 core1 */
    me_runcore(&MED2, ME_CORE_1_ID, __cluster2_core1_entry_point__);
#endif /* !sr6p3 */


    /* Load DME core */
    me_load_core(&MED3, ME_CORE_DME, __dme_code_src__, __dme_code_dst__, (uint32_t)__dme_code_len__);
    me_load_core(&MED3, ME_CORE_DME, __dme_data_src__, __dme_data_dst__, (uint32_t)__dme_data_len__);

    /* Load DSPH core */
    me_load_core(&MED3, ME_CORE_DSPH, __dsph_code_src__, __dsph_code_dst__, (uint32_t)__dsph_code_len__);
    me_load_core(&MED3, ME_CORE_DSPH, __dsph_data_src__, __dsph_data_dst__, (uint32_t)__dsph_data_len__);

    /* Start DME core (start address is ignored) */
    me_runcore(&MED3, ME_CORE_DME, 0UL);

    /* Start DSPH core (start address is ignored) */
    me_runcore(&MED3, ME_CORE_DSPH, 0UL);
}

/*
 * Example of system initialization function
 */
static void SystemInit(void)
{
    clockInit();
    siul2_init();
    siul2_start();

    BOARD_INIT_FUNC(BOARD_NAME);

    irq_init();
    gst_init(&GST_TBUD0);
    agt_init();

    /* GST driver: starting TBU_0 for AGT clock */
    gst_start(&GST_TBUD0);

    osal_init();
    osal_start();

    linflexd_init();

    /* Initialize the UART device */
    uart_init(&UARTD);

    /* Initialize Runtime IO module.*/
    io_init();

    /* Start Runtime IO module.*/
    io_start(&UARTD);
}

#if (STELLAR_AI_VALIDATE == TRUE)
/*
 * Application entry point for validation process
 */
int main(void) {

  SystemInit();

  /* Start the other cores */
  start_other_cores();

  aiValidateStart();

  /* never here...*/
  while (true) {
  }
}

#else

#include "network.h"
#include "network_data.h"

/* Global handle to reference the instantiated C-model */
STAI_NETWORK_CONTEXT_DECLARE(m_network, STAI_NETWORK_CONTEXT_SIZE)

/* Array to store the data of the activation buffers */
STAI_ALIGNED(STAI_NETWORK_ACTIVATION_1_ALIGNMENT)
static uint8_t activations_1[STAI_NETWORK_ACTIVATION_1_SIZE_BYTES];


/* Array to store the data of the input tensors */
/* -> data_in_1 is allocated in activations buffer */

/* Array to store the data of the output tensors */
/* -> data_out_1 is allocated in activations buffer */

static stai_ptr m_inputs[STAI_NETWORK_IN_NUM];
static stai_ptr m_outputs[STAI_NETWORK_OUT_NUM];
static stai_ptr m_acts[STAI_NETWORK_ACTIVATIONS_NUM];

int aiInit(void);
stai_return_code aiRun(void);
void ai_loop(void);

/*
 * Bootstrap
 */
int aiInit(void)
{
  stai_size _dummy;

  /* -- Create and initialize the c-model */

  /* Initialize the instance */
  stai_network_init(m_network);


  /* -- Set the @ of the activation buffers */

  /* Activation buffers are allocated in the user/app space */
  m_acts[0] = (stai_ptr)activations_1;
  stai_network_set_activations(m_network, m_acts, STAI_NETWORK_ACTIVATIONS_NUM);
  stai_network_get_activations(m_network, m_acts, &_dummy);



  /* -- Set the @ of the input/output buffers */

  /* Input buffers are allocated in the activations buffer */
  stai_network_get_inputs(m_network, m_inputs, &_dummy);

  /* Output buffers are allocated in the activations buffer */
  stai_network_get_outputs(m_network, m_outputs, &_dummy);

  return 0;
}

/*
 * Run inference
 */
stai_return_code aiRun()
{
  stai_return_code res;

  res = stai_network_run(m_network, STAI_MODE_SYNC);

  return res;
}

/*
 * Example of ai loop function
 */
void ai_loop() {

  aiInit();


  while (1) {
    /* 1 - Acquire, pre-process and fill the input buffers */
    // acquire_and_process_data(...);

    /* 2 - Call inference engine */
    aiRun();

    /* 3 - Post-process the predictions */
    // post_process(...);

    /* 4 - Toggle a pin at each inference */
	siul2_iotoggle(USER_LED1);
	osal_delay_milliseconds(400);
  }
}

/**
 * @brief Cluster0 Core0 main function for All core LED blink Test.
 *
 * @return void
 *
 */
int main(void) {

  SystemInit();

  /* Make sure to start with same state for all LEDs */
  siul2_ioset(USER_LED1);
  siul2_ioset(USER_LED2);
  siul2_ioset(USER_LED3);
  siul2_ioset(USER_LED4);
  siul2_ioset(USER_LED5);
  siul2_ioset(USER_LED6);
  siul2_ioset(USER_LED7);
  siul2_ioset(USER_LED8);

  /* Start the other cores */
  start_other_cores();

  printf("Hello World from Cluster0 Core0 Cortex-R52\n\r");

  /* Application main loop */
  ai_loop();

  return 0;
}
#endif /* #if (STELLAR_AI_VALIDATE == TRUE) */

/** @} */
